/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "version.h"
#include "cw_paddle_keyboard.h"
#include "ps2_protocol_keyboard.h"
#include "ps2_keyboard.h"
#include "interrupts.h"
#include "main.h"
#include <ctype.h>
#include <assert.h>

#if DEBUG == 1
#define DPUTCHAR(X) 	putchar(X);
#else
#define DPUTCHAR(X)
#endif

/******************************************************************************/
/*																			  */
/*				CW paddle Morse code character input routines				  */ 
/*																			  */		
/******************************************************************************/

/* -------------------------------------------------------------------------- */

enum {		/* parameters */

	// WORD_PER_MINUTE = 15	/* note: it goes nuts with 13 */
	WORD_PER_MINUTE = 10	/* note: it goes nuts with 13 */
};

/* -------------------------------------------------------------------------- */

enum {		/* constants */

	dit_time 	    = 1200/WORD_PER_MINUTE,		/* in milliseconds */
	letter_time     = 3*dit_time,
	word_time       = 7*dit_time,
};

enum {		/* special codes: error, spaces and command codes */

	CW_ERASE_CODE   = 0x00,
	CW_SPACE_CODE   = 0x01,
	CW_COMMAND_CODE = 0x10 + 0x03,				/* ! */
		SHIFT_KEY_CODE	   = 0x08 + 0x00,		/* !S */
		ENTER_KEY_CODE	   = 0x02 + 0x00,		/* !E */
		CW_ERASE_CODE2		   = 0x02 + 0x01,	/* !T */
		SWITCH_MOUSE_CODE	   = 0x04 + 0x03,	/* !M */
		NUE_PSK_XMIT_RECV_CODE = 0x10 + 0x09,	/* !X */
	 	SEND_VERSION_CODE 	   = 0x10 + 0x01,	/* !V */
	 	SEND_LICENSE_CODE 	   = 0x10 + 0x04,	/* !L */
		SEND_BRAG_CODE 		   = 0x10 + 0x08,	/* !B */
		SEND_SUMMARY_CODE	   = 0x40 + 0x0C,	/* !? */
};

enum {		/* useful ASCII characters */

	SPACE_CHAR		= ' ',
	BACKSPACE_CHAR  = '\b',
	ENTER_CHAR		= '\r',
	NEWLINE_CHAR	= '\n',
	INVALID_CHAR    = '*',
	HUH_CHAR		= '?',
};

/* -------------------------------------------------------------------------- */

/*
 * CW code to ASCII character translation table
 *
 *	index - CW code
 *  entry - ASCII character
 */

enum {
	CW_CODE_TABLE_SIZE = 128,     /* must be a power of 2 */
	CW_CODE_TABLE_SIZE_MASK = CW_CODE_TABLE_SIZE - 1,
};

static const char cw_code_table[CW_CODE_TABLE_SIZE] = {

	/* CW code				 ASCII character */
	/* =======				 =============== */

/* zero symbols */
   
   /* 0x00 + 0x00             */ INVALID_CHAR,		/* CW_ERASE_CODE */
   /* 0x00 + 0x01             */ INVALID_CHAR,		/* CW_SPACE_CODE */

/* one symbol */

   /* 0x02 + 0x00           . */   'e',				/* ENTER_KEY_CODE */
   /* 0x02 + 0x01           _ */   't',				/* CW_ERASE_CODE2 */

/* two symbols */

   /* 0x04 + 0x00         . . */   'i',
   /* 0x04 + 0x01         . _ */   'a',
   /* 0x04 + 0x02         _ . */   'n',
   /* 0x04 + 0x03         _ _ */   'm',

/* three symbols */

   /* 0x08 + 0x00       . . . */   's',				/* SHIFT_KEY_CODE */
   /* 0x08 + 0x01       . . _ */   'u',
   /* 0x08 + 0x02       . _ . */   'r',
   /* 0x08 + 0x03       . _ _ */   'w',
   /* 0x08 + 0x04       _ . . */   'd',
   /* 0x08 + 0x05       _ . _ */   'k',
   /* 0x08 + 0x06       _ _ . */   'g',
   /* 0x08 + 0x07       _ _ _ */   'o',

/* four symbols */

   /* 0x10 + 0x00     . . . . */   'h',				
   /* 0x10 + 0x01     . . . _ */   'v',				/* SEND_VERSION_CODE */
   /* 0x10 + 0x02     . . _ . */   'f',
   /* 0x10 + 0x03     . . _ _ */ INVALID_CHAR,		/* CW_COMMAND_CODE */
   /* 0x10 + 0x04     . _ . . */   'l',				/* SEND_LICENSE_CODE */
   /* 0x10 + 0x05     . _ . _ */ INVALID_CHAR,
   /* 0x10 + 0x06     . _ _ . */   'p',
   /* 0x10 + 0x07     . _ _ _ */   'j',

   /* 0x10 + 0x08     _ . . . */   'b',				/* SEND_BRAG_CODE */
   /* 0x10 + 0x09     _ . . _ */   'x',				/* NUE_PSK_XMIT_RECV_CODE */
   /* 0x10 + 0x0A     _ . _ . */   'c',
   /* 0x10 + 0x0B     _ . _ _ */   'y',
   /* 0x10 + 0x0C     _ _ . . */   'z',
   /* 0x10 + 0x0D     _ _ . _ */   'q',
   /* 0x10 + 0x0E     _ _ _ . */ INVALID_CHAR,
   /* 0x10 + 0x0F     _ _ _ _ */ INVALID_CHAR,

/* five symbols */

   /* 0x20 + 0x00   . . . . . */   '0',
   /* 0x20 + 0x01   . . . . _ */   '4',
   /* 0x20 + 0x02   . . . _ . */ INVALID_CHAR,
   /* 0x20 + 0x03   . . . _ _ */   '3',
   /* 0x20 + 0x04   . . _ . . */ INVALID_CHAR,
   /* 0x20 + 0x05   . . _ . _ */ INVALID_CHAR,
   /* 0x20 + 0x06   . . _ _ . */ INVALID_CHAR,
   /* 0x20 + 0x07   . . _ _ _ */   '2',

   /* 0x20 + 0x08   . _ . . . */ INVALID_CHAR,
   /* 0x20 + 0x09   . _ . . _ */ INVALID_CHAR,
   /* 0x20 + 0x0A   . _ . _ . */ INVALID_CHAR,
   /* 0x20 + 0x0B   . _ . _ _ */ INVALID_CHAR,
   /* 0x20 + 0x0C   . _ _ . . */ INVALID_CHAR,
   /* 0x20 + 0x0D   . _ _ . _ */ INVALID_CHAR,
   /* 0x20 + 0x0E   . _ _ _ . */ INVALID_CHAR,
   /* 0x20 + 0x0F   . _ _ _ _ */   '1',

   /* 0x20 + 0x10   _ . . . . */   '6',
   /* 0x20 + 0x11   _ . . . _ */   '=',
   /* 0x20 + 0x12   _ . . _ . */   '/',
   /* 0x20 + 0x13   _ . . _ _ */ INVALID_CHAR,
   /* 0x20 + 0x14   _ . _ . . */ INVALID_CHAR,
   /* 0x20 + 0x15   _ . _ . _ */ INVALID_CHAR,
   /* 0x20 + 0x16   _ . _ _ . */   '(',
   /* 0x20 + 0x17   _ . _ _ _ */ INVALID_CHAR,

   /* 0x20 + 0x18   _ _ . . . */   '7',
   /* 0x20 + 0x19   _ _ . . _ */ INVALID_CHAR,
   /* 0x20 + 0x1A   _ _ . _ . */   '+',
   /* 0x20 + 0x1B   _ _ . _ _ */ INVALID_CHAR,
   /* 0x20 + 0x1C   _ _ _ . . */   '8',
   /* 0x20 + 0x1D   _ _ _ . _ */ INVALID_CHAR,
   /* 0x20 + 0x1E   _ _ _ _ . */   '9',
   /* 0x20 + 0x1F   _ _ _ _ _ */   '0',

/* six symbols */

   /* 0x40 + 0x00 . . . . . . */ INVALID_CHAR,
   /* 0x40 + 0x01 . . . . . _ */ INVALID_CHAR,
   /* 0x40 + 0x02 . . . . _ . */ INVALID_CHAR,
   /* 0x40 + 0x03 . . . . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x04 . . . _ . . */ INVALID_CHAR,
   /* 0x40 + 0x05 . . . _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x06 . . . _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x07 . . . _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x08 . . _ . . . */ INVALID_CHAR,
   /* 0x40 + 0x09 . . _ . . _ */ INVALID_CHAR,
   /* 0x40 + 0x0A . . _ . _ . */ INVALID_CHAR,
   /* 0x40 + 0x0B . . _ . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x0C . . _ _ . . */   '?',				/* SEND_SUMMARY_CODE */
   /* 0x40 + 0x0D . . _ _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x0E . . _ _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x0F . . _ _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x10 . _ . . . . */ INVALID_CHAR,
   /* 0x40 + 0x11 . _ . . . _ */ INVALID_CHAR,
   /* 0x40 + 0x12 . _ . . _ . */   '"',
   /* 0x40 + 0x13 . _ . . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x14 . _ . _ . . */ INVALID_CHAR,
   /* 0x40 + 0x15 . _ . _ . _ */   '.',
   /* 0x40 + 0x16 . _ . _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x17 . _ . _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x18 . _ _ . . . */ INVALID_CHAR,
   /* 0x40 + 0x19 . _ _ . . _ */ INVALID_CHAR,
   /* 0x40 + 0x1A . _ _ . _ . */   '@',
   /* 0x40 + 0x1B . _ _ . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x1C . _ _ _ . . */ INVALID_CHAR,
   /* 0x40 + 0x1D . _ _ _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x1E . _ _ _ _ . */   '\'',
   /* 0x40 + 0x1F . _ _ _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x20 _ . . . . . */ INVALID_CHAR,
   /* 0x40 + 0x21 _ . . . . _ */   '-',
   /* 0x40 + 0x22 _ . . . _ . */ INVALID_CHAR,
   /* 0x40 + 0x23 _ . . . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x24 _ . . _ . . */ INVALID_CHAR,
   /* 0x40 + 0x25 _ . . _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x26 _ . . _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x27 _ . . _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x28 _ . _ . . . */ INVALID_CHAR,
   /* 0x40 + 0x29 _ . _ . . _ */ INVALID_CHAR,
   /* 0x40 + 0x2A _ . _ . _ . */ INVALID_CHAR,
   /* 0x40 + 0x2B _ . _ . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x2C _ . _ _ . . */ INVALID_CHAR,
   /* 0x40 + 0x2D _ . _ _ . _ */   ')',
   /* 0x40 + 0x2E _ . _ _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x2F _ . _ _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x30 _ _ . . . . */ INVALID_CHAR,
   /* 0x40 + 0x31 _ _ . . . _ */ INVALID_CHAR,
   /* 0x40 + 0x32 _ _ . . _ . */ INVALID_CHAR,
   /* 0x40 + 0x33 _ _ . . _ _ */   ',',
   /* 0x40 + 0x34 _ _ . _ . . */ INVALID_CHAR,
   /* 0x40 + 0x35 _ _ . _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x36 _ _ . _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x37 _ _ . _ _ _ */ INVALID_CHAR,

   /* 0x40 + 0x38 _ _ _ . . . */   ':',   			/* last entry */
   /* 0x40 + 0x39 _ _ _ . . _ */ INVALID_CHAR,
   /* 0x40 + 0x3A _ _ _ . _ . */ INVALID_CHAR,
   /* 0x40 + 0x3B _ _ _ . _ _ */ INVALID_CHAR,
   /* 0x40 + 0x3C _ _ _ _ . . */ INVALID_CHAR,
   /* 0x40 + 0x3D _ _ _ _ . _ */ INVALID_CHAR,
   /* 0x40 + 0x3E _ _ _ _ _ . */ INVALID_CHAR,
   /* 0x40 + 0x3F _ _ _ _ _ _ */ INVALID_CHAR,
};

/* -------------------------------------------------------------------------- */

static int get_cw_word(void);

static unsigned int input_cw_code(void);

static void handle_command_code(unsigned int cw_code);

static void send_str_to_host(char str[ ]);

static char *get_brag_notice(void);

static char *get_command_summary(void);

/* -------------------------------------------------------------------------- */

/*
 * this routine initializes the CW paddle keyboard routines
 */
				
void initialize_cw_paddle_keyboard(void)
{

#if TESTING == 1
	printf("initialize_cw_paddle_keyboard():\r\n");
	printf("    DEBOUNCE_DELAY:  %Ld msec\r\n"
		   "    WORD_PER_MINUTE: %d wpm\r\n"
		   "    dit_time:        %Ld msec\r\n" 
		   "    letter_time:     %Ld msec\r\n"
		   "    word_time:       %Ld msec\r\n",
		DEBOUNCE_DELAY, WORD_PER_MINUTE, dit_time, letter_time, word_time);
#endif

#if TESTING == 1
	printf("initialize_cw_paddle_keyboard() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this function is the main routine of the the CW paddle routines
 */

int cw_paddle_keyboard(void)
{
	int cw_paddle_mode;

	for (;;) {

		/* an interrupt wakes up processor */
		sleep();
		DPUTCHAR('#');

		/* a cw paddle was pressed */
		assert (get_interrupt_event() == PADDLE_INTERRUPT);
		cw_paddle_mode = get_cw_word();
		if (cw_paddle_mode != PADDLE_KEYBOARD_MODE)
			break;
	}

	return cw_paddle_mode;
}

/* -------------------------------------------------------------------------- */

/*
 * this function gets the next CW word
 */

static int get_cw_word(void)
{
	static unsigned int Cw_char_count = 0;
	static unsigned int Prev_cw_code = CW_SPACE_CODE;
	static unsigned int Cw_line_count = 0;

	unsigned int cw_code;
	unsigned char cw_char = ' ';
	int cw_paddle_mode = PADDLE_KEYBOARD_MODE;
	unsigned int i;

	for (;;) {

		cw_code = input_cw_code();

		/* six or more dits is erase word code */
		if (cw_code == CW_ERASE_CODE) {		
			DPUTCHAR('.');							 
			for (i = 0; i < Cw_char_count; i++) {
				send_char_to_host(BACKSPACE_CHAR);
				putchar('_');
			}
			Cw_char_count = 0;
			Prev_cw_code = cw_code;
			continue;
		}

		/* flush any extra dits from erase code */
		if (Prev_cw_code == CW_ERASE_CODE) {
			Prev_cw_code = cw_code;
			continue;
		}

		/* check if in command mode */
		if (cw_paddle_mode == PADDLE_COMMAND_MODE) {

			/* switch to paddle mouse mode */
			if (cw_code == SWITCH_MOUSE_CODE) {
				DPUTCHAR('!');
				cw_paddle_mode = PADDLE_MOUSE_MODE;
				break;
			}

			/* another erase word code */
			if (cw_code == CW_ERASE_CODE2) {
				DPUTCHAR('.');							 
				for (i = 0; i < Cw_char_count; i++) {
					send_char_to_host(BACKSPACE_CHAR);
					putchar('_');
				}
				cw_paddle_mode = PADDLE_KEYBOARD_MODE;
			}

			/* stay in paddle command mode while CW space codes are coming in */
			if (cw_code != CW_SPACE_CODE && cw_code != CW_ERASE_CODE2) {

				/* handle a command code */
				if (cw_code == ENTER_KEY_CODE) {
					Cw_line_count = 0;
				}
				handle_command_code(cw_code);
				cw_paddle_mode = PADDLE_KEYBOARD_MODE;

			}

			Cw_char_count = 0;
			Prev_cw_code = cw_code;
			continue;
		}

		/* check for command code */
		if (cw_code == CW_COMMAND_CODE) {
			DPUTCHAR('!');
			cw_paddle_mode = PADDLE_COMMAND_MODE;
			Prev_cw_code = cw_code;
			continue;
		} 

		if (cw_code != CW_SPACE_CODE) {
			cw_char = cw_code_table[cw_code];
		} else {
			DPUTCHAR('_');
		}

		/* check for space code */
		if (Prev_cw_code != CW_SPACE_CODE) { 
			if (cw_code != CW_SPACE_CODE) {
				send_char_to_host(cw_char);
				putchar(cw_char);
				Cw_line_count++;
				
				Cw_char_count++;
			}
		} else {
			if (cw_code != CW_SPACE_CODE) {
				if (Cw_line_count != 0 && Cw_char_count != 0) {
					send_char_to_host(SPACE_CHAR);
					putchar(SPACE_CHAR);
				}
				send_char_to_host(cw_char);
				putchar(cw_char);
				Cw_line_count++;
				Cw_char_count = 1;

			} else {
				break;		/* two spaces in sequence */
			}
		}

		cw_paddle_mode = PADDLE_KEYBOARD_MODE;
		Prev_cw_code = cw_code;
	}

	return cw_paddle_mode;
}

/* -------------------------------------------------------------------------- */

/*
 * this function returns the next CW character.  A CW character
 * is a series of dits and dahs terminated by a letter space.
 */

static unsigned int input_cw_code(void)
{
	unsigned int port_B;
	unsigned int cw_code = CW_SPACE_CODE;
	int dit_paddle;
	int dah_paddle;
	int prev_dit_paddle = FALSE;
	int prev_dah_paddle = FALSE;	

	/* wait for end of word */
	set_timer_ms(word_time);

	while (cw_code < CW_CODE_TABLE_SIZE) {

		/* port input event: a paddle was pressed */
		if (get_interrupt_event() == PADDLE_INTERRUPT) {
			DPUTCHAR('*');
			port_B = get_port_B();

			clear_interrupt_event();
			disable_interrupts(GLOBAL);

			if (bit_test(Port_B, paddle_dit_contact_bit) == LOW) {
				dit_paddle = TRUE;
			} else {
				dit_paddle = FALSE;
			} 
			if (bit_test(Port_B, paddle_dah_contact_bit) == LOW) {
				dah_paddle = TRUE;
			} else {
				dah_paddle = FALSE;
			}

			/* test for any paddle contact closed event */
			if (dit_paddle == TRUE || dah_paddle == TRUE) {

				if (dit_paddle == TRUE && prev_dit_paddle == FALSE) {
					cw_code <<= 1;
					cw_code |= 0x0;	
				}
				if (dah_paddle == TRUE && prev_dah_paddle == FALSE) {
					cw_code <<= 1;
					cw_code |= 0x1;
				}
				output_bit(STATUS_LED, TRUE);
				delay_ms(DEBOUNCE_DELAY);

			/* else both paddles have opened */
			} else {

				set_timer_ms(letter_time);
				output_bit(STATUS_LED, FALSE);
			}

			prev_dit_paddle = dit_paddle;
			prev_dah_paddle = dah_paddle;

			enable_interrupts(GLOBAL);

		/* timer3 timeout event: end of letter or word timeout */
		} else if (get_interrupt_event() == SPACE_INTERRUPT) {

			clear_interrupt_event();
			output_bit(STATUS_LED, FALSE);
			break;
		}
	}
	cw_code &= CW_CODE_TABLE_SIZE_MASK;

	/*
	 * note: CW_ERASE_CODE is generated as follows: entering six or more dits
	 * causes cw_code to be 0x80 causing the code to break out of the previous
	 * loop.  Then cw_code is masked out with 0x7F yielding a value of 0x00,
	 * the value of CW_ERASE_CODE.
	 */

	DPUTCHAR('%');
	return cw_code;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine handles a command code
 */

static void handle_command_code(unsigned int cw_code)
{
	char keyboard_str[ ] = "Keyboard: ";

	switch (cw_code) {
	case SHIFT_KEY_CODE:				/* !S */

		DPUTCHAR('S');
		toggle_caps_lock_keyboard();
		break;

	case ENTER_KEY_CODE:				/* !E */

		DPUTCHAR('E');
		send_char_to_host(ENTER_CHAR);
		putchar(ENTER_CHAR);
		putchar(NEWLINE_CHAR);
		break;

	case NUE_PSK_XMIT_RECV_CODE:		/* !X */

		DPUTCHAR('X');
		send_F10_keyboard();
		break;

	case SEND_VERSION_CODE:				/* !V */

		DPUTCHAR('V');
		printf("%s\r\n", get_version());
		send_char_to_host(ENTER_CHAR);
		send_str_to_host(get_version());
		send_char_to_host(ENTER_CHAR);

		printf("%s\r\n", get_copyright_notice());
		send_str_to_host(get_copyright_notice());
		send_char_to_host(ENTER_CHAR);
		send_char_to_host(ENTER_CHAR);
		break;

	case SEND_LICENSE_CODE:				/* !L */

		DPUTCHAR('L');
		printf("%s\r\n", get_version());
		send_char_to_host(ENTER_CHAR);
		send_str_to_host(get_version());
		send_char_to_host(ENTER_CHAR);

		printf("%s\r\n", get_copyright_notice());
		send_str_to_host(get_copyright_notice());
		send_char_to_host(ENTER_CHAR);
		send_char_to_host(ENTER_CHAR);

		send_str_to_host(get_license_notice());
		send_char_to_host(ENTER_CHAR);
		send_char_to_host(ENTER_CHAR);
		break;

	case SEND_BRAG_CODE:				/* !B */

		DPUTCHAR('B');
		send_char_to_host(ENTER_CHAR);
		send_str_to_host(get_brag_notice());
		send_char_to_host(ENTER_CHAR);

		send_str_to_host(keyboard_str);
		send_str_to_host(get_version());
		send_char_to_host(ENTER_CHAR);
		send_char_to_host(ENTER_CHAR);
		break;

	case SEND_SUMMARY_CODE:				/* !? */

		DPUTCHAR('?');
		send_char_to_host(ENTER_CHAR);
		send_str_to_host(get_command_summary());
		send_char_to_host(ENTER_CHAR);
		send_char_to_host(ENTER_CHAR);
		break;

	default:
		send_char_to_host(HUH_CHAR);
		delay_ms(500);
		send_char_to_host(BACKSPACE_CHAR);
		break;
	}
}

/* -------------------------------------------------------------------------- */

/*
 * send a string to host
 */

static void send_str_to_host(char str[ ])
{
	unsigned long i = 0;

	while (str[i] != '\0') {
		send_char_to_host(str[i++]);
		delay_ms(200);
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the brag notice
 */

// temporary: implement this with EEPROM library routines

static char *get_brag_notice(void)
{
	static char str[ ] =
		"Name: DAVE, delta alpha victor echo\r"
		"QTH: Potomac, Montgomery County, Maryland, USA\r"
		"Locator: FM19ja\r"
		"Antenna: homebrew 20 meter vertical\r"
		"Rig: NUE-PSK PSK31 modem with Small Wonder Labs PSK-20";
		
	return str;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the command summary
 */

static char *get_command_summary(void)
{
	static char str[ ] =
		"using ! to represent the command code . . _ _\r"
		"   !S send CAPS LOCK\r"
		"   !E send ENTER\r"
		"   !T erase word\r"
		"   !M switch to mouse mode\r"
		"   !X send F10 key (for NUE-PSK modem)\r"
		"   !V send program version\r"
		"   !L send license info\r"
		"   !B send brag info\r"
		"   !? send this summary";

	return str;
}

/* -------------------------------------------------------------------------- */

/*
 * this routines demonstrates the PS/2 keyboard by sending a string
 */

void demo_ps2_keyboard(void)
{
	printf("demonstrate PS/2 keyboard emulation\r\n");

	send_char_to_host(ENTER_CHAR);
	send_str_to_host(get_version());
	send_char_to_host(ENTER_CHAR);
}

/* -------------------------------------------------------------------------- */

#if TESTING == 1
/* ========================================================================== */
/*																			  */
/*								TEST ROUTINES								  */
/*																			  */			
/* ========================================================================== */

/*
 * this routine is the tests cw paddle text input
 */

void test_cw_paddle_keyboard(void)
{

	blink_status_led(1);
	printf("set a text editor as the active window\r\n");
	delay_ms(1000);

	cw_paddle_keyboard();	/* ignore return value */
}

/* -------------------------------------------------------------------------- */

#endif

